globals [ 
  week
  week-list
  
  role-names
  
  initial-inventory     ;how much beer do brewers / wholesalers / retailers start with?
  initial-orders-queue 
  initial-shipping-queue
  initial-production-queue
  initialized?          ;have the agents been initialized?
  
  inventory-fee
  backlog-fee

  demand-list 
  consumer-demand
  
  plot-variable
  
  ;; quick start instructions variables
  quick-start  ;; current quickstart instruction displayed in the quickstart monitor
  qs-item      ;; index of the current quickstart instruction
  qs-items     ;; list of quickstart instructions
]

breeds [
  retailers
  wholesalers
  distributors
  factories  
  ]

turtles-own [
  user-id                ;login name of the student associated with that turtle
  my-plot

  client                 ;client associated with that turtle
  supplier               ;supplier for that turtle
  role                   ;role number played by that turtle
  team                   ;a number shared by all related turtles - factory, distributor, wholesaler, retailer
  
  shipment-received
  inventory              ;amount of beer stored away  
  backlog                ;how much was the turtle short?  CUMULATIVE
  inventory-paper        ;inventory - backlog
  shipment-sent
  order-received
  order-placed
  current-order          ;the order to send FROM inventory
  next-order
  
  inventory-cost          ;how much did inventory cost this week?  
  backlog-cost
  cost
  
  inventory-record 
  backlog-record 
  inventory-paper-record
  received-record 
  order-received-record 
  shipped-record 
  order-placed-record 

  cost-record 
  inventory-cost-record 
  backlog-cost-record 
  
  ledger
  ]  

factories-own [
  production-delay
  ]

to startup
  setup-quick-start
  hubnet-set-client-interface "COMPUTER" [ "clients/Beer Game client.nlogo" ]
  hubnet-reset
  setup true
end

to setup [ full-setup? ]
  if full-setup?
  [
    cg
    clear-output
    set role-names ["factory" "distributor" "wholesaler" "retailer"]
    set inventory-fee .5
    set backlog-fee 1
    set demand-list sentence [4 4 4 4 4] n-values 32 [8]
  ]
  
  clear-all-plots
  
  set week 0
  set week-list []
  set initialized? false

  if not full-setup?
  [ initialize-turtles ]
end

to initialize-turtles
  ;check to see if there are enough players
  set initialized? false
  if not (count turtles mod 4 = 0)
  [ user-message "Sorry... you need four players per team."
    stop
  ]
  
  ask turtles
  [ set size 1
    set color gray
    set inventory 12
    set backlog 0

    set inventory-cost 0
    set backlog-cost 0
    set cost 0

    set order-received 4
    set order-placed 4
    set shipment-received 4
    set shipment-sent 4
    set current-order 4
    set next-order 4
   
    set inventory-record []
    set backlog-record []
    set inventory-paper-record []
    set cost-record []
    set received-record []
    set order-received-record []
    set shipped-record []
    set order-placed-record []
    set inventory-cost-record []
    set backlog-cost-record []

    set ledger ""
    set my-plot ""
        
    update-client-display
  ]

  ask factories
  [ set production-delay 4]
  
  update-plot
  set initialized? true
end

;; give the user some information about what the setup button does so they can 
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? ("The SETUP button should only be used when starting "
              + "over with a new group (such as a new set of students) since "
              + "all data is lost.  Use the RE-RUN button for continuing with "
              + "an existing group."
              + "\n\nDo you really want to setup the model?")
  [ setup true ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;        Runtime Procedures
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
  ;the first time it is run, initialize the queues
  if not initialized? [initialize-turtles]  
  ;stops the procedure if the initialization didn't work... for example, if there aren't even four players  
  if not initialized? [stop]
  
  ;iterate weeks
  every 50 / model-speed
  [ if week > 35 [stop]
    set week week + 1
    set week-list lput week week-list

    ;determine consumer demand
    set consumer-demand item week demand-list
    
    ;STEP 1: move shipments received to inventory
    ask turtles [set inventory inventory + shipment-received]
    ;STEP 2: move shipment-sent to client's shipment-received
    ask turtles with [not (role = 0)]
    [ set shipment-received shipment-sent-of supplier]
    ask factories
    [ set shipment-received production-delay]
    ;STEP 3: move inventory to shipment-sent
    ask turtles
    [ ifelse current-order + backlog <= inventory
      [ set shipment-sent current-order + backlog ]
      [ set shipment-sent inventory ]
      set inventory inventory - shipment-sent
      set backlog backlog + (current-order - shipment-sent)
    ]
    ;STEP 4: determine the current order (for next week)
    ask turtles
    [ set current-order order-received ]
    ;STEP 5: move orders placed to orders received
    ask turtles with [not (role = 3)]
    [ set order-received order-placed-of client]
    ask factories 
    [ set production-delay order-placed]
    ask retailers
    [ set order-received consumer-demand]
    ;STEP 6: place orders
    ask turtles
    [ set order-placed next-order ]
    
    ;charge for storage
    ask turtles [ set inventory-cost precision (inventory-fee * inventory) 2 ]
    ;charge for backlog
    ask turtles [ set backlog-cost precision (backlog-fee * backlog) 2 ]
    ;figure the total cost
    ask turtles [ set cost inventory-cost + backlog-cost]
    
    ask turtles [ set inventory-paper (inventory - backlog) ]

    ;fill memory lists
    record-data

    ;update displays
    ask turtles [update-client-display]
    update-graphics-window    
    update-plot   
    update-my-plots   
  ]  
end

to record-data
  set week-list lput week week-list
  ask turtles
  [ set inventory-record lput inventory inventory-record
    set backlog-record lput backlog backlog-record
    set inventory-paper-record lput inventory-paper inventory-paper-record
    set cost-record lput cost cost-record

    set received-record lput shipment-received received-record
    set order-received-record lput order-received order-received-record
    set shipped-record lput shipment-sent shipped-record
    set order-placed-record lput order-placed order-placed-record
    set inventory-cost-record lput inventory-cost inventory-cost-record
    set backlog-cost-record lput backlog-cost backlog-cost-record
    
    if week mod 10 = 1
    [ set ledger (ledger + "WEEK  SHIPMENTS  SHIPMENTS  NET CHANGE  EFFECTIVE  WEEKLY  CUMULATIVE  ORDERS\n"
                         + "      RECEIVED   SENT       INVENTORY   INVENTORY  COST    COST        SENT\n")
    ] 
    set ledger (ledger 
              + ensure-string-length 4 week + "  "
              + ensure-string-length 9 shipment-received + "  "
              + ensure-string-length 9 shipment-sent + "  "
              + ensure-string-length 10 (shipment-received - shipment-sent) + "  "
              + ensure-string-length 9 (inventory - backlog) + "  "
              + ensure-string-length 6 cost + "  "
              + ensure-string-length 10 sum cost-record + "  "
              + ensure-string-length 6 order-placed + "\n")
    if week mod 10 = 0
    [ set ledger ledger + "\n"
    ]
  ]
end

;; report a string version of some-input that has a length of exactly string-width
to-report ensure-string-length [string-width some-input]
  let string-out "" + some-input
  ifelse length string-out > string-width
  [
    ifelse is-number? some-input
    [ set string-out "" + precision (string-width - 2) some-input ]
    [ set string-out substring string-out 0 string-width ]
  ]
  [
    if length string-out < string-width
    [
      repeat string-width - length string-out
      [ set string-out string-out + " " ]
    ]
  ]
  report string-out
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;        Interacting with turtles through clicks procedures
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; listen for a mouse click to see if we should show information about a particular turtle
to listen-click
  if mouse-down? and any? turtles-at mouse-xcor mouse-ycor
  [ pop-up random-one-of turtles-at mouse-xcor mouse-ycor ]
end

;; show in a pop-up dialog information regarding a chosen turtle.  allow the user to plot
;; this information.
to pop-up [chosen-turtle]
  let option user-choice ("Player: " + user-id-of chosen-turtle + "\n"
                        + "Role: " + item role-of chosen-turtle role-names + "\n"
                        + "Team: " + team-of chosen-turtle + "\n"
                        + "Inventory: " + inventory-of chosen-turtle + "\n"
                        + "Backlog: " + backlog-of chosen-turtle + "\n"
                        + "Team Inventory: " + sum values-from turtles with [team = team-of chosen-turtle] [inventory] + "\n"
                        + "Team Backlog: " + sum values-from turtles with [team = team-of chosen-turtle] [backlog] + "\n"
                        ) ["Continue" "Plot 1" "Plot 2"]
  ifelse option = "Plot 1"
  [ setup-and-fill-turtle-plot "plot1" chosen-turtle ]
  [
    ifelse option = "Plot 2"
    [ setup-and-fill-turtle-plot "plot2" chosen-turtle ]
    [] ;; we don't do anything if the user chose "Continue"
  ]
end

;; clear and setup the plot with the name turtle-plot-name.  fill it with the current information.
to setup-and-fill-turtle-plot [ turtle-plot-name chosen-turtle ]
  ;; use without-interruption so that we can safely call this procedure from a different button 
  ;; than the button that is running the plotting code
  without-interruption
  [
    set-current-plot turtle-plot-name
    clear-plot
    ask turtles with [ my-plot = turtle-plot-name ]
    [ set my-plot ""]
    ask chosen-turtle
    [
      set my-plot turtle-plot-name
      fill-my-plot
    ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;        plotting procedures
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to listen-plot-widgets
  if not (plot-variable = global-plot )
  [ set plot-variable global-plot
    fill-plot  
  ]
end

;; this procedure must be called from a method which sets 
;; the current plot and uses without-interruption so that 
;; multiple buttons with plotting code can run simultaneously.
to fill-my-plot  ;; turtle procedure
  let index 0
  repeat week
  [ set-current-plot-pen "zero"
    plot 0

    set-current-plot-pen "Inventory"
    plot item index inventory-paper-record
    set-current-plot-pen "Orders Sent"
    plot item index order-placed-record 
    set-current-plot-pen "Shipments Received"
    plot item index received-record 
    set-current-plot-pen "Orders Received"
    plot item index order-received-record 
    set-current-plot-pen "Shipments Sent"
    plot item index shipped-record 
    set-current-plot-pen "Cost"
    plot item index cost-record 
 
    set index index + 1
  ]
end

to update-my-plots
 without-interruption
 [ ask turtles with [not (my-plot = "")]
   [  set-current-plot my-plot
      set-current-plot-pen "zero"
      plot 0

      set-current-plot-pen "Inventory"
      plot inventory-paper
      set-current-plot-pen "Orders Sent"
      plot order-placed 
      set-current-plot-pen "Shipments Received"
      plot shipment-received 
      set-current-plot-pen "Orders Received"
      plot order-received 
      set-current-plot-pen "Shipments Sent"
      plot shipment-sent 
      set-current-plot-pen "Cost"
      plot cost      
    ]
  ] 
end

to fill-plot
  let code-to-execute ""
  let code-hash ( list
    [ "None"
      false ]
    [ "Team Cost - Cumulative"
      "sum values-from turtles with [team = team-of myself] [sum-items 0 index cost-record]" ]
    [ "Team Cost - Weekly"
      "sum values-from turtles with [team = team-of myself] [item index cost-record]" ]
    [ "Team Inventory"
      "sum values-from turtles with [team = team-of myself] [item index inventory-record]" ]
    [ "Retailer Cost - Cumulative"
      "sum-items 0 index cost-record-of one-of retailers with [team = team-of myself]" ]
    [ "Retailer Cost - Weekly"
      "item index cost-record-of one-of retailers with [team = team-of myself]" ]
    [ "Retailer Inventory"
      "item index inventory-record-of random-one-of retailers with [team = team-of myself]" ] )

  without-interruption
  [
    set-current-plot "Beer-Game Info"
    clear-plot
  
    let index 0
    repeat week
    [ set-current-plot-pen "zero"
      plot 0
      
      ask factories
      [ set-current-plot-pen team-pen team
        ;; figure out what plotting metric code we are supposed to execute
        foreach code-hash [ if first ? = plot-variable [ set code-to-execute last ? ] ]
        ;; ensure that we are supposed to execute some code for the current value 
        ;; of plot-variable, as well as ensuring that it is safe to execute the code
        if code-to-execute != false and __check-syntax ("plot " + code-to-execute) = ""
        [ plot run-result code-to-execute ]
      ]
      set index index + 1
    ]
  ]
end

to-report sum-items [start finish some-list]
  let index start
  let some-sum 0
  repeat (finish - start + 1)
  [ set some-sum some-sum + item index some-list
    set index index + 1
  ] 
  report some-sum
end

to update-plot
  let code-to-execute ""
  let code-hash ( list
    [ "None"
      false ]
    [ "Team Cost - Cumulative"
      "sum values-from turtles with [team = team-of myself] [sum cost-record]" ]
    [ "Team Cost - Weekly"
      "sum values-from turtles with [team = team-of myself] [cost]" ]
    [ "Team Inventory"
      "sum values-from turtles with [team = team-of myself] [inventory]" ]
    [ "Retailer Cost - Cumulative"
      "sum cost-record-of random-one-of retailers with [team = team-of myself]" ]
    [ "Retailer Cost - Weekly"
      "cost-of random-one-of retailers with [team = team-of myself]" ]
    [ "Retailer Inventory"
      "inventory-of random-one-of retailers with [team = team-of myself]" ] )
  
  without-interruption
  [
    set-current-plot "Beer-Game Info"
    set-current-plot-pen "zero"
    plot 0
    
    ask factories
    [
      set-current-plot-pen team-pen team
      ;; figure out what plotting metric code we are supposed to execute
      foreach code-hash [ if first ? = plot-variable [ set code-to-execute last ? ] ]
      ;; ensure that we are supposed to execute some code for the current value 
      ;; of plot-variable, as well as ensuring that it is safe to execute the code
      if code-to-execute != false and __check-syntax ("plot " + code-to-execute) = ""
      [ plot run-result code-to-execute ]
    ]
  ]
end

to-report team-pen [team-number]
  if team-number = 1 [report "teamOne"]
  if team-number = 2 [report "teamTwo"]
  if team-number = 3 [report "teamThree"]
  if team-number = 4 [report "teamFour"]
  if team-number = 5 [report "teamFive"]
  if team-number = 6 [report "teamSix"]
  if team-number = 7 [report "teamSeven"]
  if team-number = 8 [report "teamEight"]
  if team-number = 9 [report "teamNine"]
  if team-number = 10 [report "teamTen"]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;        graphics window procedures
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to update-graphics-window
  if player-size = "None"
  [ ask turtles [ set size 1]]
  if player-size = "Cost"
  [ ask turtles [ size-cost]]
  if player-size = "Inventory"
  [ ask turtles [ size-inventory]]
  
  if player-color = "None"
  [ ask turtles [ set color gray]]
  if player-color = "Cost"
  [ ask turtles [ color-cost]]
  if player-color = "Inventory"
  [ ask turtles [ color-inventory]]      
end

to size-cost  ;; turtle procedure
  ifelse cost <= 2 
  [ set size .5]
  [ ifelse cost <= 5 
    [ set size .75]
   [ ifelse cost <= 10
      [ set size 1]
      [ ifelse cost <= 20
        [ set size 1.5]
        [ ifelse cost <= 30
          [ set size 2]
          [ set size 2.5]
        ]
      ]
    ]
  ] 
end

to size-inventory  ;; turtle procedure
  ifelse inventory = 0
  [ set size .5]
  [ ifelse inventory <= 6 
    [ set size .75]
    [ ifelse inventory <= 12
      [ set size 1]
      [ ifelse inventory <= 18
        [ set size 1.5]
        [ ifelse inventory <= 24
          [ set size 2]
          [ set size 2.5]
        ]
      ]
    ]
  ] 
end

to color-cost  ;; turtle procedure
  ifelse cost <= 2 
  [ set color (blue + 2)]
  [ ifelse cost <= 5 
    [ set color blue]
   [ ifelse cost <= 10
      [ set color (blue - 2)]
      [ ifelse cost <= 20
        [ set color (red - 2)]
        [ ifelse cost <= 30
          [ set color red ]
          [ set color (red + 2)]
        ]
      ]
    ]
  ] 
end

to color-inventory  ;; turtle procedure
  ifelse inventory = 0
  [ set color red + 2]
  [ ifelse inventory <= 6 
    [ set color red]
    [ ifelse inventory <= 12
      [ set color red + 2]
      [ ifelse inventory <= 18
        [ set color blue + 2]
        [ ifelse inventory <= 24
          [ set color blue ]
          [ set color blue - 2]
        ]
      ]
    ]
  ] 
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;        Adding agents to the simulation
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to add-agent [id]
  if count turtles >= 40
  [ user-message "Sorry... the distribution simulation can only support up to 40 participants."
    stop
  ]
 
  ;if there are no turtles, or all current turtles have all the clients they can handle, then make a new brewer
  ifelse (not any? turtles with [(not (role = 3)) and (client = nobody)])
  [ create-factory id ]
  [ ;if there are turtles who have no client, 
    ;then pick one of them to make a new client for themselves.
    ask random-one-of turtles with [(not (role = 3)) and (client = nobody)] 
    [ create-new-player id (role + 1)]
  ]    
end

to create-factory [id]
  create-custom-factories 1
  [ set role 0
    set shape "factory"
    set color gray
    set heading 90
    set client nobody
    set supplier nobody
    set user-id id
    setxy (- screen-edge-x + 1) (screen-edge-y + 1 - 2 * count factories)
    set team count factories
    ask patch-at -1 1 
    [ set plabel team-of myself ]
  ]
end

to create-new-player [id role-type]
  hatch 1
  [ set role role-type
    if role = 1
    [ set breed distributors
      set shape "distributor"
    ]
    if role = 2
    [ set breed wholesalers
      set shape "wholesaler"
    ]
    if role = 3
    [ set breed retailers
      set shape "retailer"
    ] 
    set color gray
    set client nobody
    ;set the supplier for this turtle as the brewer who called the hatch command
    set supplier myself
    set user-id id
    ;; this next line is confusing.  The first myself is the agent doing the 
    ;; hatching.  The second myself is the hatched agent.  So, the line reads:
    ;; "ask the agent who is doing the hatch to set its client variable to the 
    ;; newly hatched agent."
    ask myself [set client myself]
    fd 2          
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;        Processing Information from the Clients
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message? 
    [
      ;; we don't want clients connecting while the game is being played.
      if not initialized? 
      [ on-user-connect hubnet-message-source ]
    ]
    [
      ifelse hubnet-exit-message?
      [ on-user-exit hubnet-message-source ]
      [ if hubnet-message-tag = "^" [ increase-order ]
        if hubnet-message-tag = "v" [ decrease-order ]
      ]
    ]
  ]
end

to on-user-connect [id]
  add-agent id
  ask turtles with [user-id = id]
  [ hubnet-send user-id "User-ID" user-id
    hubnet-send user-id "Role" item role role-names
    hubnet-send user-id "Team" team
    ifelse role = 0
    [ hubnet-send user-id "Supplier" "N/A" ]  ;; factories don't have suppliers
    [
      hubnet-send user-id "Supplier" user-id-of supplier
      ;; notify my supplier that i am it's client
      hubnet-send user-id-of supplier "Client" user-id
    ]
    ifelse role = 3
    [ hubnet-send user-id "Client" "N/A"]  ;; retailers don't have clients
    [ hubnet-send user-id "Client" "waiting"]  ;; we don't know who our client is yet
    hubnet-send user-id "Storage Fee" inventory-fee
    hubnet-send user-id "Backlog Fee" backlog-fee
  ]  
end

to on-user-exit [id]
  ask turtles with [user-id = id]
  [ die ]
end

to increase-order
  ask turtles with [user-id = hubnet-message-source] 
  [ set next-order next-order + 1
    hubnet-send user-id "Next Order To Place" next-order
  ]
end

to decrease-order
  ask turtles with [user-id = hubnet-message-source] 
  [ set next-order next-order - 1
    if next-order < 0 [set next-order 0]
    hubnet-send user-id "Next Order To Place" next-order
  ]
end

to update-client-display  ;; turtle procedure
  hubnet-send user-id "Week" week

  hubnet-send user-id "<- Client Demand" current-order
  hubnet-send user-id "<- Order Placed" order-placed
  hubnet-send user-id "Shipment Sent ->" shipment-sent
  hubnet-send user-id "Inventory" inventory
  hubnet-send user-id "Backlog" backlog
  hubnet-send user-id "Shipment Received ->" shipment-received
  
  hubnet-send user-id "Weekly Inv. $" inventory-cost
  hubnet-send user-id "Total Inv. $" sum inventory-cost-record  
  hubnet-send user-id "Weekly Back $" backlog-cost
  hubnet-send user-id "Total Back $" sum backlog-cost-record
  hubnet-send user-id "Weekly Cost $" cost
  hubnet-send user-id "Total Cost" sum cost-record

  hubnet-send user-id "Next Order To Place" next-order

  hubnet-send user-id "Ledger" ledger
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, and clients to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to run the HubNet activity."
      "Optional: Zoom In (see Tools in the Menu Bar)"
      "Optional: Change any of the settings...."
      "If you do change the settings, press the SETUP button."
      "Press the RUN button."
      "Note: There must be a multiple of 4 students logged in when the simulation starts."
    "Everyone: Open up a HubNet Client on your machine and..."
      "input the IP Address of this computer and..."
      "type your user name in the User Name box and press ENTER"
    "Teacher: Have the students acquaint themselves with the interface."
      "Explain how the game works, i.e. the rules for the how beer is bought and sold,..."
      "that there is a delay between ordering a shipment and receiving it, and..."
      "how costs accrue."
      "Once the students are familiar with the client interface and understand the rules,..."
      "press the NetLogo GO button to start the simulation."
    "Everyone: Help your team minimize its costs."
    "Teacher: While the simulation is running, you can change various aspects of..."
      "the simulation:"
      "To change the speed of the simulation, change the MODEL-SPEED slider."
      "To change the metric which determines how the turtles are colored and sized,..."
      "change the value of the PLAYER-COLOR and PLAYER-SIZE choices."
      "To change the metric that is shown in the BEER-GAME INFO plot, change..."
      "the value of the GLOBAL-PLOT choice."
      "To use the student plots, click on a turtle in the Graphics Window and..."
      "choose the PLOT 1 or PLOT 2 buttons."
    
    "Teacher: To rerun the activity with the same group,..."
      "stop the model by pressing the NetLogo GO button, if it is on."
      "Change any of the settings that you would like."
      "Press the NetLogo RE-RUN button."
      "Restart the simulation by pressing the NetLogo GO button again."
    
    "Teacher: To start the simulation over with a new group,..."
      "stop the model by pressing the NetLogo GO and RUN buttons, if they are on."
      "Press the SETUP button and follow these instructions again from the beginning."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end


; *** NetLogo Model Copyright Notice ***
;
; This activity and associated models and materials was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and 
; INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; These projects gratefully acknowledge the support of the 
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2003 by Uri Wilensky. Updated 2003. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution 
; for profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2003).  NetLogo HubNet Beer Game model.
; http://ccl.northwestern.edu/netlogo/models/HubNetBeerGame.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 1998 by Uri Wilensky.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/HubNetBeerGame
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
4
55
187
447
5
11
15.73
1
10
1
1
1
0

CC-WINDOW
5
541
782
636
Command Center

CHOOSER
195
281
287
326
player-size
player-size
"None" "Cost" "Inventory"
2

CHOOSER
195
328
287
373
player-color
player-color
"None" "Cost" "Inventory"
1

BUTTON
4
10
77
51
Setup
setup-prompt
NIL
1
T
OBSERVER
T
NIL

BUTTON
121
10
194
51
Re-Run
setup false
NIL
1
T
OBSERVER
T
NIL

BUTTON
195
107
287
140
Go
go
T
1
T
OBSERVER
T
NIL

MONITOR
295
95
398
144
Week
"week " + week
0
1

BUTTON
195
10
287
51
RUN
listen-clients\nlisten-click \nlisten-plot-widgets
T
1
T
OBSERVER
NIL
NIL

MONITOR
195
230
287
279
Who is it?
user-id-of random-one-of turtles-at mouse-xcor mouse-ycor
0
1

SLIDER
195
140
287
173
model-speed
model-speed
1
10
5
1
1
NIL

PLOT
295
146
773
352
Beer-Game Info
time
value
0.0
30.0
0.0
30.0
true
true
PENS
"zero" 1.0 0 -7566196 true
"teamOne" 1.0 0 -16777216 true
"teamTwo" 1.0 0 -16776961 true
"teamThree" 1.0 0 -65536 true
"teamFour" 1.0 0 -11352576 true
"teamFive" 1.0 0 -44544 true
"teamSix" 1.0 0 -8716033 true
"teamSeven" 1.0 0 -6524078 true
"teamEight" 1.0 0 -7566196 true
"teamNine" 1.0 0 -16711681 true
"teamTen" 1.0 0 -65281 true

CHOOSER
602
99
773
144
global-plot
global-plot
"None" "Team Cost - Cumulative" "Team Cost - Weekly" "Team Inventory" "Retailer Cost - Cumulative" "Retailer Cost - Weekly" "Retailer Inventory"
1

BUTTON
195
64
287
105
One Week
go
NIL
1
T
OBSERVER
T
NIL

PLOT
295
359
527
527
plot1
week
value
0.0
30.0
0.0
30.0
true
false
PENS
"zero" 1.0 0 -7566196 true
"Inventory" 1.0 0 -16777216 true
"Orders Sent" 1.0 0 -16776961 true
"Shipments Received" 1.0 0 -65536 true
"Orders Received" 1.0 0 -11352576 true
"Shipments Sent" 1.0 0 -44544 true
"Cost" 1.0 0 -8716033 true

PLOT
536
359
773
527
plot2
week
value
0.0
30.0
0.0
30.0
true
false
PENS
"zero" 1.0 0 -7566196 true
"Inventory" 1.0 0 -16777216 true
"Orders Sent" 1.0 0 -16776961 true
"Shipments Received" 1.0 0 -65536 true
"Orders Received" 1.0 0 -11352576 true
"Shipments Sent" 1.0 0 -44544 true
"Cost" 1.0 0 -8716033 true

BUTTON
295
60
413
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

BUTTON
631
60
715
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
554
60
632
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
T
NIL

MONITOR
295
10
715
59
Quick Start Instructions- More in Info Window
quick-start
0
1

@#$#@#$#@
WHAT IS IT?
-----------
This is a NetLogo adaptation of the MIT Beer Game.  In it, students play members of a beer distribution network (factory, distributor, wholesaler, or retailer).  They place and ship orders, trying to minimize their costs.


HOW IT WORKS
------------
Each week, a member receives orders for a certain number of cases of beer from downstream.  They fill these orders as well as they can (and if unable to fill the orders, form a backlog), and place their own orders.  There is both a two week order delay (so, two weeks pass between placing an order and the person upstream receiving it) and a two week shipping delay (two weeks pass between sending an order out the door and the client receiving it).  Participants also pay $.50 a week for each case of inventory, and $1 a week for each case of backlog.


QUICKSTART INSTRUCTIONS
-----------------------
Teacher: Follow these directions to run the HubNet activity.
Optional: Zoom In (see Tools in the Menu Bar)
Optional: Change any of the settings.  If you do change the settings, press the SETUP button.
Press the RUN button. Note: There must be a multiple of 4 students logged in when the simulation starts.

Everyone: Open up a HubNet Client on your machine and input the IP Address of this computer and type your user name in the User Name box and press ENTER.

Teacher: Have the students acquaint themselves with the interface.  Explain how the game works, i.e. the rules for the how beer is bought and sold, that there is a delay between ordering a shipment and receiving it, and how costs accrue.  Once the students are familiar with the client interface and understand the rules, press the NetLogo GO button to start the simulation.

Everyone: Help your team minimize its costs.

Teacher: While the simulation is running, you can change various aspects of the simulation:
--To change the speed of the simulation, change the MODEL-SPEED slider.
--To change the metric which determines how the turtles are colored and sized, change the value of the PLAYER-COLOR and PLAYER-SIZE choices.
--To change the metric that is shown in the BEER-GAME INFO plot, change the value of the GLOBAL-PLOT choice.
--To use the student plots, click on a turtle in the Graphics Window and choose the PLOT 1 or PLOT 2 buttons.

Teacher: To rerun the activity with the same group, stop the model by pressing the NetLogo GO button, if it is on.  Change any of the settings that you would like.  Press the NetLogo RE-RUN button.  Restart the simulation by pressing the NetLogo GO button again.

Teacher: To start the simulation over with a new group, stop the model by pressing the NetLogo GO and RUN buttons, if they are on.  Press the SETUP button and follow these instructions again from the beginning.


HOW TO USE IT
-------------
It is important that the RUN button be on the entire time.  This button enables the clients to login and adjust their orders, and for the interface elements (like clicking on turtles or changing the plots) to work.

When the model starts, it is already set up.  After that, hit the RUN button for students to login using the HubNet clients.  It is important that the students login in groups of four!  If a group is incomplete, the model will not run.

Take some time to explain the interface elements to the students, both on the teacher model and the HubNet clients.  Use the ONE WEEK button to show the students what happens after one week of the sim.  Note that the students are sending and receiving cases of beer according to built-in pre-orders - the system begins in equilibrium.

If students don't notice it, call their attention to their costs.  Point out that they are charged $.50 for each case of beer they have in inventory each week... but that they are charged $1 for every case that is ordered from downstream, but which they don't have!  If they build up a backlog, they will have to try to eventually fill those orders.  

Once the students are comfortable with the interface and the cost rules, go ahead and hit the Re-Run button.  This will allow you to start over from the beginning, and set the students loose on the challenge - for each team to minimize its costs!

Tell the students that the challenge will run for 50 weeks (though it will actually only run 36... this is to avoid students making decisions influenced by the impending end).  You can then use the GO or ONE WEEK buttons to run the simulation (GO will automatically run through the weeks, and ONE WEEK will step through the weeks one at a time).  At the end of the time, keep the RUN button on to be able to look at different information during class discussion.

When the RUN button is down, you can choose what data is displayed in the upper plot by using the GLOBAL-PLOT chooser.  You can view each team's current and cumulative costs, and their current inventory.  You can also view each retailer's costs and inventory.  The lower plots can display information for individual turtles.  Click on a turtle to call up a menu with display options.


THINGS TO NOTICE
----------------
Notice how backlogs and over-inventories tend to come in waves.  


THINGS TO TRY
-------------
Challenge students to develop strategies to minimize their costs immediately after explaining the rules of the system.  Then test these strategies, and try to revise them.  

Turn off the up-front display, so students can only see how they are doing as an individual.  How does this relate to real business situations?


EXTENDING THE MODEL
-------------------
Add a second interface area on the right for the teacher to build a custom customer demand function.


CREDITS AND REFERENCES
----------------------
For a discussion of how to use the beer game, what issues to raise in discussions, and the importance of systems dynamics, see http://web.mit.edu/jsterman/www/SDG/beergame.html

For a bibliography of the beer game in academic publications (last revised July 1992), see http://www.sol-ne.org/repository/download/bibl.html?item_id=456327

Instructions for running the paper based beer game:
http://www.sol-ne.org/repository/download/instr.html?item_id=456354

This activity and associated models and materials was created as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and INTEGRATED SIMULATION AND MODELING ENVIRONMENT. These projects gratefully acknowledge the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227.

Copyright 2003 by Uri Wilensky. Updated 2003. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:
a) this copyright notice is included.
b) this model will not be redistributed for profit without permission from the copyright holders.
Contact the copyright holders for appropriate licenses for redistribution for profit.

To refer to this model in academic publications, please use: Wilensky, U. (2003). NetLogo HubNet Beer Game model. http://ccl.northwestern.edu/netlogo/models/HubNetBeerGame. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL. 

In other publications, please use: Copyright 2003 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetBeerGame for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

distributor
false
0
Polygon -7566196 true true 75 28 75 255 209 255 255 209 262 178 262 104 256 59 209 29 119 29 120 67 205 67 226 81 226 174 200 212 119 212 121 27

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

factory
false
0
Rectangle -7566196 true true 17 155 287 285
Rectangle -7566196 true true 29 105 137 155
Rectangle -7566196 true true 225 16 246 155
Rectangle -7566196 true true 180 16 201 155

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

retailer
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

wholesaler
false
0
Polygon -7566196 true true 36 60 89 264 114 264 150 196 184 261 210 261 265 58 220 58 189 171 150 117 108 164 83 59

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
